# Design Document

## Overview

This document describes the technical design for migrating the MyGB school management system from SQLite to MySQL/MySQLi. The migration will be implemented in a phased approach to minimize risk and ensure data integrity. The design maintains backward compatibility during the transition period and provides clear rollback mechanisms.

### Migration Strategy

The migration follows a **dual-database support pattern** during transition:
1. Phase 1: Add MySQL support alongside existing SQLite
2. Phase 2: Implement data migration tools
3. Phase 3: Switch default to MySQL
4. Phase 4: (Optional) Remove SQLite support

This approach allows testing and validation before fully committing to MySQL.

## Architecture

### High-Level Architecture

```
┌─────────────────────────────────────────────────────────┐
│                   MyGB Application                       │
├─────────────────────────────────────────────────────────┤
│                                                          │
│  ┌──────────────────────────────────────────────────┐  │
│  │         Database Abstraction Layer               │  │
│  │         (includes/db.php)                        │  │
│  │                                                   │  │
│  │  ┌──────────────┐      ┌──────────────┐        │  │
│  │  │   SQLite     │      │    MySQL     │        │  │
│  │  │   Driver     │      │   Driver     │        │  │
│  │  └──────────────┘      └──────────────┘        │  │
│  └──────────────────────────────────────────────────┘  │
│                      │              │                   │
└──────────────────────┼──────────────┼───────────────────┘
                       │              │
                       ▼              ▼
              ┌─────────────┐  ┌─────────────┐
              │   SQLite    │  │    MySQL    │
              │  school.db  │  │ mygb_school │
              └─────────────┘  └─────────────┘
```

### Database Connection Flow

```
Application Request
       │
       ▼
┌──────────────────┐
│  get_db()        │
│  function        │
└──────────────────┘
       │
       ▼
┌──────────────────┐
│ Read config      │
│ database.php     │
└──────────────────┘
       │
       ├─── DB_TYPE = 'mysql' ──┐
       │                         │
       └─── DB_TYPE = 'sqlite'  │
                                 │
                                 ▼
                    ┌────────────────────────┐
                    │  Create MySQLi         │
                    │  Connection            │
                    │  - Host: DB_HOST       │
                    │  - User: DB_USER       │
                    │  - Pass: DB_PASS       │
                    │  - DB: DB_NAME         │
                    └────────────────────────┘
                                 │
                                 ▼
                    ┌────────────────────────┐
                    │  Wrap in PDO-like      │
                    │  Interface             │
                    └────────────────────────┘
                                 │
                                 ▼
                         Return Connection
```

## Components and Interfaces

### 1. Configuration Component

**File:** `config/database.php`

**Purpose:** Centralized database configuration

**Structure:**
```php
<?php
// Database Configuration
define('DB_TYPE', 'mysql'); // 'mysql' or 'sqlite'

// MySQL Configuration (used when DB_TYPE = 'mysql')
define('DB_HOST', 'localhost');
define('DB_NAME', 'mygb_school');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_CHARSET', 'utf8mb4');
define('DB_PORT', 3306);

// SQLite Configuration (used when DB_TYPE = 'sqlite')
define('SQLITE_DB_PATH', __DIR__ . '/../data/school.db');
?>
```

**Template File:** `config/database.php.example`
- Provides example configuration
- Includes comments explaining each setting
- Used for fresh installations

### 2. Database Abstraction Layer

**File:** `includes/db.php`

**Purpose:** Provide unified database interface supporting both SQLite and MySQL

**Key Functions:**

#### `get_db(): PDO|MySQLiPDOWrapper`
- Returns database connection based on configuration
- Implements singleton pattern for connection reuse
- Handles connection errors gracefully

#### `MySQLiPDOWrapper` Class
- Wraps MySQLi in PDO-compatible interface
- Provides `prepare()`, `exec()`, `query()` methods
- Translates between MySQLi and PDO result formats
- Handles parameter binding compatibility

**Implementation Strategy:**
```php
function get_db() {
    static $db = null;
    if ($db !== null) return $db;
    
    if (!file_exists(__DIR__ . '/../config/database.php')) {
        die('Configuration file missing. Please copy config/database.php.example to config/database.php');
    }
    
    require_once __DIR__ . '/../config/database.php';
    
    if (DB_TYPE === 'mysql') {
        $db = create_mysql_connection();
    } else {
        $db = create_sqlite_connection();
    }
    
    return $db;
}
```

### 3. Migration Script Component

**File:** `migrate_to_mysql.php`

**Purpose:** Automated data migration from SQLite to MySQL

**Migration Process Flow:**

```
Start Migration
      │
      ▼
┌─────────────────┐
│ Pre-flight      │
│ Checks          │
│ - MySQL running │
│ - Credentials   │
│ - SQLite exists │
└─────────────────┘
      │
      ▼
┌─────────────────┐
│ Backup SQLite   │
│ Database        │
└─────────────────┘
      │
      ▼
┌─────────────────┐
│ Create MySQL    │
│ Database        │
└─────────────────┘
      │
      ▼
┌─────────────────┐
│ Create Schema   │
│ - Tables        │
│ - Indexes       │
│ - Constraints   │
└─────────────────┘
      │
      ▼
┌─────────────────┐
│ Migrate Data    │
│ Table by Table  │
└─────────────────┘
      │
      ▼
┌─────────────────┐
│ Verify Data     │
│ - Count check   │
│ - Sample verify │
└─────────────────┘
      │
      ▼
┌─────────────────┐
│ Generate Report │
│ - Success/Fail  │
│ - Record counts │
│ - Warnings      │
└─────────────────┘
      │
      ▼
   Complete
```

**Key Features:**
- Transaction support for rollback on failure
- Progress indicators for large datasets
- Detailed logging to `migration_log.txt`
- Dry-run mode for testing
- Automatic backup before migration

### 4. Schema Converter Component

**Purpose:** Convert SQLite schema to MySQL-compatible schema

**Conversion Rules:**

| SQLite Type | MySQL Type | Notes |
|-------------|------------|-------|
| `INTEGER PRIMARY KEY AUTOINCREMENT` | `INT AUTO_INCREMENT PRIMARY KEY` | Auto-increment ID |
| `TEXT` (short) | `VARCHAR(255)` | For names, titles |
| `TEXT` (medium) | `TEXT` | For descriptions |
| `TEXT` (long) | `LONGTEXT` | For content |
| `REAL` | `DECIMAL(10,2)` | For currency |
| `BLOB` | `BLOB` | For binary data |
| `CURRENT_TIMESTAMP` | `CURRENT_TIMESTAMP` | Datetime default |

**SQLite-specific Syntax Conversions:**

| SQLite | MySQL |
|--------|-------|
| `INSERT OR IGNORE` | `INSERT IGNORE` |
| `INSERT OR REPLACE` | `REPLACE INTO` |
| `PRAGMA foreign_keys = ON` | (Enabled by default in InnoDB) |
| `PRAGMA table_info(table)` | `SHOW COLUMNS FROM table` |
| `AUTOINCREMENT` | `AUTO_INCREMENT` |

### 5. Backup Module Updates

**File:** `admin/database_backup.php`

**MySQL Backup Strategy:**

**Option 1: PHP-based Export**
- Read all tables and data
- Generate SQL INSERT statements
- Suitable for small to medium databases

**Option 2: mysqldump Integration**
- Use system `mysqldump` command
- Faster for large databases
- Requires shell_exec permission

**Implementation:**
```php
function backup_mysql_database($outputFile) {
    $host = DB_HOST;
    $user = DB_USER;
    $pass = DB_PASS;
    $db = DB_NAME;
    
    // Try mysqldump first
    if (function_exists('shell_exec')) {
        $command = sprintf(
            'mysqldump -h%s -u%s -p%s %s > %s',
            escapeshellarg($host),
            escapeshellarg($user),
            escapeshellarg($pass),
            escapeshellarg($db),
            escapeshellarg($outputFile)
        );
        shell_exec($command);
        if (file_exists($outputFile)) {
            return true;
        }
    }
    
    // Fallback to PHP export
    return php_export_mysql($outputFile);
}
```

### 6. Initialization Script Updates

**File:** `init_db.php`

**Purpose:** Initialize fresh database (MySQL or SQLite)

**MySQL Initialization Process:**
1. Connect to MySQL server
2. Create database if not exists
3. Select database
4. Create all tables with proper schema
5. Insert default settings
6. Create default admin account
7. Create upload directories

**Schema Definitions:**

```sql
-- Students Table (MySQL)
CREATE TABLE IF NOT EXISTS students (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nis VARCHAR(50) UNIQUE NOT NULL,
    name VARCHAR(255) NOT NULL,
    grade VARCHAR(50),
    photo VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_nis (nis),
    INDEX idx_grade (grade)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Admins Table (MySQL)
CREATE TABLE IF NOT EXISTS admins (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_username (username)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Settings Table (MySQL)
CREATE TABLE IF NOT EXISTS settings (
    `key` VARCHAR(100) PRIMARY KEY,
    value TEXT,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Additional tables (slides, menu, berita, etc.)
-- Follow same pattern with proper MySQL types
```

## Data Models

### Student Model

**SQLite Schema:**
```sql
CREATE TABLE students (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    nis TEXT UNIQUE,
    name TEXT NOT NULL,
    grade TEXT,
    photo TEXT,
    created_at TEXT DEFAULT CURRENT_TIMESTAMP
)
```

**MySQL Schema:**
```sql
CREATE TABLE students (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nis VARCHAR(50) UNIQUE NOT NULL,
    name VARCHAR(255) NOT NULL,
    grade VARCHAR(50),
    photo VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_nis (nis),
    INDEX idx_grade (grade)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
```

**Field Mappings:**
- `id`: Auto-increment primary key
- `nis`: Student ID number (unique)
- `name`: Full name
- `grade`: Class/grade (e.g., "10A", "11 IPA 2")
- `photo`: Filename of uploaded photo
- `created_at`: Registration timestamp

### Admin Model

**MySQL Schema:**
```sql
CREATE TABLE admins (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
```

**Security Considerations:**
- Passwords stored using `password_hash()` with bcrypt
- Username must be unique
- No plain text password storage

### Settings Model

**MySQL Schema:**
```sql
CREATE TABLE settings (
    `key` VARCHAR(100) PRIMARY KEY,
    value TEXT,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
```

**Common Settings:**
- `site_name`: School name
- `site_description`: School description
- `site_logo`: Logo filename
- `site_address`: Physical address
- `site_phone`: Contact phone
- `site_email`: Contact email

## Error Handling

### Connection Errors

**MySQL Connection Failure:**
```php
try {
    $mysqli = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME, DB_PORT);
    if ($mysqli->connect_error) {
        throw new Exception("MySQL Connection Failed: " . $mysqli->connect_error);
    }
} catch (Exception $e) {
    // Log error
    error_log($e->getMessage());
    
    // Display user-friendly message
    die('
        <h1>Database Connection Error</h1>
        <p>Unable to connect to MySQL database.</p>
        <h3>Troubleshooting:</h3>
        <ul>
            <li>Verify MySQL server is running</li>
            <li>Check database credentials in config/database.php</li>
            <li>Ensure database "' . DB_NAME . '" exists</li>
            <li>Verify user has proper permissions</li>
        </ul>
    ');
}
```

### Query Errors

**Prepared Statement Errors:**
```php
try {
    $stmt = $db->prepare($sql);
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $db->error);
    }
    $stmt->execute($params);
} catch (Exception $e) {
    error_log("Query Error: " . $e->getMessage());
    error_log("SQL: " . $sql);
    error_log("Params: " . print_r($params, true));
    
    // User-friendly message
    flash_set('error', 'Database operation failed. Please try again.');
}
```

### Migration Errors

**Rollback Strategy:**
```php
function migrate_with_transaction() {
    $mysql->begin_transaction();
    
    try {
        // Perform migration steps
        migrate_students();
        migrate_admins();
        migrate_settings();
        
        // Verify data
        if (!verify_migration()) {
            throw new Exception("Data verification failed");
        }
        
        $mysql->commit();
        return true;
    } catch (Exception $e) {
        $mysql->rollback();
        log_error("Migration failed: " . $e->getMessage());
        return false;
    }
}
```

## Testing Strategy

### Unit Testing

**Database Connection Tests:**
- Test MySQL connection with valid credentials
- Test MySQL connection with invalid credentials
- Test SQLite connection fallback
- Test configuration file missing scenario

**Query Compatibility Tests:**
- Test INSERT operations
- Test SELECT with WHERE clauses
- Test UPDATE operations
- Test DELETE operations
- Test JOIN queries
- Test LIMIT and OFFSET

### Integration Testing

**Migration Tests:**
- Test full migration with sample data
- Test migration with empty database
- Test migration with large dataset (1000+ records)
- Test migration rollback on failure
- Test data integrity after migration

**Application Tests:**
- Test student CRUD operations
- Test admin authentication
- Test settings management
- Test file uploads
- Test backup/restore functionality

### Manual Testing Checklist

**Pre-Migration:**
- [ ] Backup SQLite database
- [ ] Verify MySQL server running
- [ ] Test MySQL credentials
- [ ] Check disk space

**During Migration:**
- [ ] Monitor migration progress
- [ ] Check for error messages
- [ ] Verify no data loss warnings

**Post-Migration:**
- [ ] Login to admin panel
- [ ] View student list
- [ ] Add new student
- [ ] Edit existing student
- [ ] Delete student
- [ ] Update settings
- [ ] Upload logo
- [ ] Create database backup
- [ ] Test public pages

### Performance Testing

**Benchmarks to Compare:**
- Page load time (SQLite vs MySQL)
- Query execution time for common operations
- Concurrent user handling (10, 50, 100 users)
- Large dataset queries (1000+ records)

**Expected Improvements with MySQL:**
- Better concurrent access (no database locking)
- Faster complex queries with proper indexing
- Better scalability for production use

## Security Considerations

### Configuration Security

**Protect database.php:**
```apache
# .htaccess in config/
<Files "database.php">
    Require all denied
</Files>
```

**Environment Variables (Optional):**
```php
// Use environment variables for sensitive data
define('DB_USER', getenv('MYSQL_USER') ?: 'root');
define('DB_PASS', getenv('MYSQL_PASS') ?: '');
```

### SQL Injection Prevention

**Always use prepared statements:**
```php
// GOOD - Prepared statement
$stmt = $db->prepare('SELECT * FROM students WHERE nis = ?');
$stmt->execute([$nis]);

// BAD - String concatenation
$sql = "SELECT * FROM students WHERE nis = '$nis'"; // NEVER DO THIS
```

### Password Security

**Use PHP password functions:**
```php
// Hashing
$hash = password_hash($password, PASSWORD_DEFAULT);

// Verification
if (password_verify($password, $hash)) {
    // Login successful
}
```

## Deployment Strategy

### Phase 1: Preparation (Week 1)
- Create configuration template
- Update database abstraction layer
- Add MySQL support alongside SQLite
- Test on development environment

### Phase 2: Migration Tools (Week 2)
- Develop migration script
- Test with sample data
- Create documentation
- Prepare rollback procedures

### Phase 3: Testing (Week 3)
- Comprehensive testing
- Performance benchmarking
- Security audit
- User acceptance testing

### Phase 4: Production Migration (Week 4)
- Schedule maintenance window
- Backup production database
- Run migration script
- Verify data integrity
- Monitor for issues
- Keep SQLite backup for 30 days

### Rollback Plan

If issues occur after migration:

1. **Immediate Rollback:**
   - Change `DB_TYPE` to 'sqlite' in config
   - Restore SQLite backup if needed
   - Application continues with SQLite

2. **Investigate Issues:**
   - Review migration logs
   - Check data integrity
   - Identify root cause

3. **Fix and Retry:**
   - Address identified issues
   - Test fix in staging
   - Schedule new migration attempt

## Documentation Requirements

### User Documentation

**MYSQL_MIGRATION_GUIDE.md:**
- Prerequisites and requirements
- Step-by-step migration instructions
- Configuration guide
- Troubleshooting common issues
- FAQ section

### Developer Documentation

**Code Comments:**
- Document all new functions
- Explain complex logic
- Note MySQL-specific code

**API Documentation:**
- Database abstraction layer API
- Migration script usage
- Configuration options

## Maintenance and Monitoring

### Post-Migration Monitoring

**Monitor for 30 days:**
- Database connection errors
- Query performance issues
- Data integrity problems
- User-reported issues

### Regular Maintenance

**Weekly:**
- Check database size
- Review error logs
- Monitor query performance

**Monthly:**
- Optimize tables
- Update indexes
- Review backup strategy

**Quarterly:**
- Security audit
- Performance review
- Capacity planning
